/**
  ******************************************************************************
  * @file    cp.h 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    July 14, 2016
  * @brief   master include file for the sample communication protocol
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  ******************************************************************************  
  */ 


#ifndef CP_H
#define CP_H

#ifdef SIMULATE_LOOPBACK
#define RUN_AS_SERVER
#define RUN_AS_CLIENT
#endif

// this is to test different sequential number generation algorithms, see comment in cp_frame.c

#ifdef RUN_AS_SERVER
#define TEST_SEQUENTIAL_NUMBERS
#endif

#define CP_AUTHSTATUS_FLAG_AUTHRECEIVED (1 << 0)
#define CP_AUTHSTATUS_FLAG_PEERMUSTAUTHENTICATE (1 << 1)

#define MSGPUMPQUEUEDEPTH 20

typedef enum 
{
    CP_CMD_SEND_SINGLE_PAYLOAD_TLV,
    CP_CMD_TERMINATE_CONNECTION
}   CP_COMMANDS;

typedef enum
{
    TAG_TELEGID = 0,
    TAG_RESPONSE,
    TAG_AUTHREQUEST = 0x10,
    TAG_RESCHANGENOTIFICATION = 0x80,
    TAG_RESCHANGESET,
    TAG_MAX = 0xffffffff
} CP_ALLOWEDTAGS;

typedef struct CP_MINTAGSTRUCT
{
    unsigned long m_Tag;
    unsigned long m_Length;
    unsigned char m_Value[];
} CP_MINTAGSTRUCT, *PCP_MINTAGSTRUCT;

typedef enum
{
    TELEG_SCHEDULED = 0,
    TELEG_SENT,
    TELEG_ACKED
} TELEGSTATUS;

#include "cp_frame.h"
#include "cp_priqueue.h"

typedef struct CP_MSGPUMPINFO
{
    CP_COMMANDS m_Cmd;
    unsigned long m_SeqNo;
    CP_PRILEVELS m_Pri;
    union
    {
        struct
        {
            unsigned long m_Tag;
            unsigned long m_Value;
            TELEGSTATUS m_Status;
            TickType_t m_ScheduleTime;
        } singleTLVpayloaddata;
    } m_u;
} CP_MSGPUMPINFO, *PCP_MSGPUMPINFO;

#define RECEIVERTASKPRIORITY (configMAX_PRIORITIES-2)
#define RECEIVERTASKSTACKSIZE configMINIMAL_STACK_SIZE*4

#define COMMTASKPRIORITY (configMAX_PRIORITIES-2)


typedef struct CP_PROCESSOR
{
    CP_FSM_IN m_FSMStruct;
    CP_PRIQUEUEHANDLE m_CommQueue;
    xSemaphoreHandle m_WriterMutex;
    TaskHandle_t m_AssociatedReceiverTask;
    CP_INCOMINGSINGLETLVPACKETCALLBACK m_SingleTLVCallback;
    int m_Socket;
    unsigned long m_AuthStatus;
    unsigned long m_CurrentSeqNr;
} CP_PROCESSOR, *PCP_PROCESSOR;

#include "cp_tag.h"

// For demonstration purposes, this authentication process simply sends amagic number in the auth packet from the client to the server. In real life, the
// client should provide credentials to the server to examine. This process can become arbitrarily elaborate up to the point of negotiating a session key
// and using crypted communication thereafter to prevent playback attacks.

#define CP_MAGIC_NUMBER_AUTHENTICATION 0x54321678


extern CP_STATUSCODE cp_HandleEstablishedComm(PCP_PROCESSOR p_CommInfo);                          

extern CP_STATUSCODE cp_RespondPacket(unsigned long p_SequenceNo,unsigned long p_ResponseCode, PCP_PROCESSOR p_Processor);
extern CP_STATUSCODE cp_SendSingleIntegerTagPacket(unsigned long p_SequenceNo,unsigned long p_Tag,unsigned long p_TagValue, PCP_PROCESSOR p_Processor);

extern CP_STATUSCODE cp_CommLoopServer(unsigned short p_LocalPort,PCP_PROCESSOR p_CommInfo);
extern CP_STATUSCODE cp_CommLoopClient(unsigned long p_Ipv4Address,unsigned short p_PeerPort,PCP_PROCESSOR p_CommInfo);

// public identifiers - everything below here is visible to applications!

// this is an opaque handle, meaning that its internal structure is not relevant to the caller. An application obtains this handle via cpInitCommLayer<Client|Server> and
// must pass the handle unmodified to the functions that require a handle. No assumptions whatsoever may be made about the handle and its meaning.

typedef PCP_PROCESSOR CP_COMMHANDLE; 

extern CP_COMMHANDLE cp_initCommLayerServer(void);
extern CP_COMMHANDLE cp_initCommLayerClient(void);
extern CP_STATUSCODE cp_QueuePacketToComm(CP_COMMHANDLE p_CommHandle,unsigned long p_Tag,unsigned long p_Value,CP_PRILEVELS p_Pri);
extern CP_STATUSCODE cp_RegisterIncomingSingleTLVPacketCallback(CP_COMMHANDLE p_CommHandle,CP_INCOMINGSINGLETLVPACKETCALLBACK p_Callback);

extern CP_STATUSCODE CP_FrameAndEmitPacket(unsigned char *p_Packet,unsigned long p_PacketLen,PCP_PROCESSOR p_Processor);

extern CP_STATUSCODE CP_PrepareInBuf(PCP_FSM_IN p_ProtInFSM);
extern CP_STATUSCODE CP_ReleaseInBuf(PCP_FSM_IN p_ProtInFSM);
extern CP_STATUSCODE CP_DispatchInPacketToUpperLayer(unsigned char *p_DataBuf,unsigned long p_DataLen,PCP_PROCESSOR p_Processor);

extern CP_STATUSCODE CP_ProcessResponsePacket(CP_DYNTAGSTRUCT *p_DynTagStruct,PCP_PROCESSOR p_Processor);
extern CP_STATUSCODE CP_MarkPacketAsSentAndAwaitingACK(PCP_MSGPUMPINFO p_CurrentComm,PCP_PROCESSOR p_CommInfo);

extern unsigned long CP_NextSeqNo(CP_COMMHANDLE p_CommHandle);

extern void CP_ProcessInChar(unsigned char p_Char,PCP_PROCESSOR p_Processor);

extern CP_STATUSCODE CP_EmitCharVerbatim(unsigned char p_Char,PCP_PROCESSOR p_Processor);
extern CP_STATUSCODE CP_EmitChunkVerbatim(unsigned char *p_Chunk,unsigned long p_ChunkLen,PCP_PROCESSOR p_Processor);

extern CP_STATUSCODE CP_PeerIsAuthenticated(PCP_PROCESSOR p_Processor);

#endif

